# IBBL Gateway Integration PHP Library

This repository provides a PHP library to integrate with the Islami Bank Bangladesh Limited (IBBL) Payment Gateway. The library supports functionalities for **checkout**, **status inquiry**, and **refunds** using IBBL's REST API.

---

## Features

- **Secure Payment Processing**: Initiate payment transactions with ease.
- **Transaction Status Inquiry**: Check the status of a transaction.
- **Refund Management**: Process refunds for transactions.
- **Sandbox Mode Support**: Test integration in a development environment.

---

## Installation

Use [Composer](https://getcomposer.org/) to install the library:

```bash
composer require revoltify/ibbl-payment-gateway
```

---

## Getting Started

Here is a sample implementation of the IBBL Gateway integration:

### Initialize the Gateway

```php
require 'vendor/autoload.php';

use IBBL\IBBLGateway;

$gateway = new IBBLGateway(
    clientId: 'IBB.MRCNT.727242343536',
    merchantSecret: '4b4w453456866f2345989346346b8e1ewetet2ab766sf',
    username: 'R3stwqerwetwetUs3r',
    password: 'R3stT3stqwrettUs3rP@ss',
    sandboxMode: true // Set to false in production
);
```

---

### Perform a Checkout

```php
use IBBL\Params\CheckoutParams;
use IBBL\Constants\PaymentMethods;

$params = CheckoutParams::setAmount('10') // Set transaction amount
    ->setProductName('Test Product') // Name of the product/service
    ->setPaymentMethod(PaymentMethods::MCASH) // Payment method (e.g., MCASH)
    ->setClientRefId(uniqid()) // Unique reference ID for the transaction
    ->setReturnUrl('https://your-domain.com/success') // Return URL after payment
    ->setEComInvokeSite('your-domain.com'); // E-commerce site invoking the payment

$response = $gateway->checkout($params);

// Get the redirect URL for the payment page
echo $response->getRedirectUrl();
```

---

### Check Transaction Status

```php
$response = $gateway->status('40ae99b650982d52ed71f4cqerwett7b1b3d40b2ed202a233e79dbc948c70468b6');

// View transaction details as an array
var_dump($response->toArray());
```

---

### Process a Refund

```php
use IBBL\Params\RefundParams;

$params = RefundParams::setAmount('10') // Refund amount
    ->setToken('c649040b31wqretertc1697537eb17ddd4f32793f2aef79f4dfd3f6bac6a5dc4b55') // Transaction token
    ->setRefundReason('Customer Requested Refund') // Reason for the refund
    ->setClientRefId('674ec327e1598'); // Unique reference ID for the refund

$response = $gateway->refund($params);

// View refund details as an array
var_dump($response->toArray());
```

---

### Error Handling

All methods throw an `IBBLException` in case of errors. You can catch and handle these exceptions as follows:

```php
use IBBL\Exception\IBBLException;

try {
    // Your integration logic here
} catch (IBBLException $e) {
    echo 'Error: ' . $e->getMessage();
}
```

---

## Environment Setup

1. **Sandbox Mode**: Use `sandboxMode: true` for testing. Ensure sandbox credentials are provided.
2. **Production Mode**: Set `sandboxMode: false` and use live credentials.