<?php

namespace GenixCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Typography;
use \Elementor\Control_Media;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Xotric Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TG_Hero_Banner extends Widget_Base
{

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'hero-banner';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Hero Banner', 'genixcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'genix-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['genixcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['genixcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {

        // layout Panel
        $this->start_controls_section(
            'genix_layout',
            [
                'label' => esc_html__('Design Layout', 'genixcore'),
            ]
        );
        $this->add_control(
            'tg_design_style',
            [
                'label' => esc_html__('Select Layout', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'genixcore'),
                    'layout-2' => esc_html__('Layout 2', 'genixcore'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();


        // Banner
        $this->start_controls_section(
            'tg_banner_area',
            [
                'label' => esc_html__('Banner Area', 'genixcore'),
            ]
        );

        $this->add_control(
            'banner_title',
            [
                'label' => esc_html__('Main Title', 'genixcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Creative Digital Agency for Mindful Brands', 'genixcore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'banner_desc',
            [
                'label' => esc_html__('Description', 'genixcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Lorem ipsum dolor sit amet consectetur. Ut tellus suspendisse nulla aliquam. Risus rutrum facilisis vulputate cursus vivamus.', 'genixcore'),
                'label_block' => true,
            ]
        );

        $this->end_controls_section();


        // Banner Images
        $this->start_controls_section(
            '__tg_banner_images',
            [
                'label' => esc_html__('Banner Images', 'genixcore'),
            ]
        );

        $this->add_control(
            'banner_img',
            [
                'label' => esc_html__('Choose Image', 'genixcore'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'bg_shape_color',
            [
                'label' => esc_html__('Background Shape Color', 'genixcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .banner-area::after' => 'background: {{VALUE}}',
                ],
                'condition' => [
                    'tg_design_style' => 'layout-1'
                ]
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'repeater_animation',
            [
                'label' => esc_html__('Animation Delay(ms)', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('.2', 'genixcore'),
            ]
        );

        $repeater->add_control(
            'repeater_img',
            [
                'label' => esc_html__('Choose Shapes', 'genixcore'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'banner_img_lists',
            [
                'label' => esc_html__('Image Lists', 'genixcore'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'repeater_img' => [
                            'url' => \Elementor\Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'repeater_img' => [
                            'url' => \Elementor\Utils::get_placeholder_image_src(),
                        ],
                    ],
                ],
                'condition' => [
                    'tg_design_style' => 'layout-2'
                ]
            ]
        );

        $this->end_controls_section();


        // Banner Shapes
        $this->start_controls_section(
            '__tg_banner_shapes',
            [
                'label' => esc_html__('Background Shapes', 'genixcore'),
            ]
        );

        $this->add_control(
            'shape_img',
            [
                'label' => esc_html__('Choose Shapes', 'genixcore'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'shape_img2',
            [
                'label' => esc_html__('Choose Shapes 02', 'genixcore'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'shape_img3',
            [
                'label' => esc_html__('Choose Shapes 03', 'genixcore'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'tg_design_style!' => 'layout-2'
                ]
            ]
        );

        $this->end_controls_section();


        // Banner Button
        $this->start_controls_section(
            '__tg_banner_button',
            [
                'label' => esc_html__('Banner Button', 'genixcore'),
            ]
        );

        $this->add_control(
            'banner_button_show',
            [
                'label' => esc_html__('Show Button', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'banner_btn_text',
            [
                'label' => esc_html__('Button Text', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Get Started', 'genixcore'),
                'title' => esc_html__('Enter button text', 'genixcore'),
                'label_block' => true,
                'condition' => [
                    'banner_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'banner_btn_link_type',
            [
                'label' => esc_html__('Button Link Type', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
                'condition' => [
                    'banner_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'banner_btn_link',
            [
                'label' => esc_html__('Button link', 'genixcore'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'genixcore'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'banner_btn_link_type' => '1',
                    'banner_button_show' => 'yes'
                ],
                'label_block' => true,
            ]
        );

        $this->add_control(
            'banner_btn_page_link',
            [
                'label' => esc_html__('Select Button Page', 'genixcore'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => genix_get_all_pages(),
                'condition' => [
                    'banner_btn_link_type' => '2',
                    'banner_button_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();

        // Video Button
        $this->start_controls_section(
            '__tg_video_button',
            [
                'label' => esc_html__('Video Button', 'genixcore'),
                'condition' => [
                    'tg_design_style' => 'layout-2'
                ]
            ]
        );

        $this->add_control(
            'video_text',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => esc_html__('Video Button Text', 'genixcore'),
                'default' => esc_html__('See How Its Works', 'genixcore'),
                'placeholder' => esc_html__('Type video text here', 'genixcore'),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'video_url',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => esc_html__('Video URL', 'genixcore'),
                'default' => esc_html__('https://www.youtube.com/watch?v=bixR-KIJKYM', 'genixcore'),
                'placeholder' => esc_html__('Type video url here', 'genixcore'),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->end_controls_section();


        // TAB_STYLE
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style', 'genixcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'banner_padding',
            [
                'label' => esc_html__('Padding', 'genixcore'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .tg-section-area' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'text_transform',
            [
                'label' => esc_html__('Text Transform', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => esc_html__('None', 'genixcore'),
                    'uppercase' => esc_html__('UPPERCASE', 'genixcore'),
                    'lowercase' => esc_html__('lowercase', 'genixcore'),
                    'capitalize' => esc_html__('Capitalize', 'genixcore'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // Style Tab Here
        $this->start_controls_section(
            '_section_style_content',
            [
                'label' => esc_html__('Typography', 'genixcore'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        // Title
        $this->add_control(
            '_heading_title',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__('Title', 'genixcore'),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'title_spacing',
            [
                'label' => esc_html__('Bottom Spacing', 'genixcore'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .title' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Text Color', 'genixcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .title' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'title',
                'selector' => '{{WRAPPER}} .title',
            ]
        );

        // description
        $this->add_control(
            '_content_description',
            [
                'type' => Controls_Manager::HEADING,
                'label' => esc_html__('Description', 'genixcore'),
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'description_spacing',
            [
                'label' => esc_html__('Bottom Spacing', 'genixcore'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .tg-content p' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'description_color',
            [
                'label' => esc_html__('Text Color', 'genixcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tg-content p' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'description',
                'selector' => '{{WRAPPER}} .tg-content p',
            ]
        );

        $this->end_controls_section();
    }


    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display(); ?>


        <?php if ($settings['tg_design_style'] == 'layout-2') :

            // Link
            if ('2' == $settings['banner_btn_link_type']) {
                $this->add_render_attribute('tg-button-arg', 'href', get_permalink($settings['banner_btn_page_link']));
                $this->add_render_attribute('tg-button-arg', 'target', '_self');
                $this->add_render_attribute('tg-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('tg-button-arg', 'class', 'btn btn-two wow fadeInLeft');
                $this->add_render_attribute('tg-button-arg', 'data-wow-delay', '.2s');
            } else {
                if (!empty($settings['banner_btn_link']['url'])) {
                    $this->add_link_attributes('tg-button-arg', $settings['banner_btn_link']);
                    $this->add_render_attribute('tg-button-arg', 'class', 'btn btn-two wow fadeInLeft');
                    $this->add_render_attribute('tg-button-arg', 'data-wow-delay', '.2s');
                }
            }

        ?>

            <section class="banner-area-two tg-section-area">
                <div class="container">
                    <div class="row align-items-center">
                        <div class="col-xl-7 col-lg-6">
                            <div class="banner-contant-two tg-content tg-heading-subheading animation-style3">

                                <?php if (!empty($settings['banner_title'])) : ?>
                                    <h2 class="title tg-element-title"><?php echo genix_kses($settings['banner_title']); ?></h2>
                                <?php endif; ?>

                                <?php if (!empty($settings['banner_desc'])) : ?>
                                    <p class="wow fadeInUp" data-wow-delay=".4s"><?php echo genix_kses($settings['banner_desc']); ?></p>
                                <?php endif; ?>

                                <div class="banner-btn">
                                    <?php if (!empty($settings['banner_button_show'])) : ?>
                                        <a <?php echo $this->get_render_attribute_string('tg-button-arg'); ?>>
                                            <?php echo $settings['banner_btn_text']; ?><span></span>
                                        </a>
                                    <?php endif; ?>

                                    <?php if (!empty($settings['video_text'])) : ?>
                                        <a href="<?php echo esc_url($settings['video_url']); ?>" class="play-btn popup-video wow fadeInRight" data-wow-delay=".2s">
                                            <i class="fas fa-play"></i>
                                            <?php echo $settings['video_text']; ?>
                                        </a>
                                    <?php endif; ?>

                                </div>

                            </div>
                        </div>

                        <?php if (!empty($settings['banner_img']['url'])) : ?>
                            <div class="col-xl-5 col-lg-6">
                                <div class="banner-img-two">
                                    <img src="<?php echo esc_url($settings['banner_img']['url']); ?>" alt="<?php echo esc_attr__('Banner Image', 'genixcore') ?>" class="wow fadeInUp" data-wow-delay=".2s">
                                    <ul class="list-wrap">
                                        <?php foreach ($settings['banner_img_lists'] as $item) : ?>
                                            <li><img src="<?php echo esc_url($item['repeater_img']['url']); ?>" alt="<?php echo esc_attr__('Shape', 'genixcore') ?>" class="wow slideInDown" data-wow-delay="<?php echo esc_attr($item['repeater_animation']) ?>s"></li>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                            </div>
                        <?php endif; ?>

                    </div>
                </div>
                <div class="banner-shape-wrap-two">
                    <?php if (!empty($settings['shape_img']['url'])) : ?>
                        <img src="<?php echo esc_url($settings['shape_img']['url']); ?>" alt="<?php echo esc_attr__('Banner Shape', 'genixcore') ?>">
                    <?php endif; ?>

                    <?php if (!empty($settings['shape_img2']['url'])) : ?>
                        <img src="<?php echo esc_url($settings['shape_img2']['url']); ?>" alt="<?php echo esc_attr__('Banner Shape', 'genixcore') ?>">
                    <?php endif; ?>
                </div>
            </section>

        <?php else :

            // Link
            if ('2' == $settings['banner_btn_link_type']) {
                $this->add_render_attribute('tg-button-arg', 'href', get_permalink($settings['banner_btn_page_link']));
                $this->add_render_attribute('tg-button-arg', 'target', '_self');
                $this->add_render_attribute('tg-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('tg-button-arg', 'class', 'btn wow fadeInUp');
                $this->add_render_attribute('tg-button-arg', 'data-wow-delay', '.6s');
            } else {
                if (!empty($settings['banner_btn_link']['url'])) {
                    $this->add_link_attributes('tg-button-arg', $settings['banner_btn_link']);
                    $this->add_render_attribute('tg-button-arg', 'class', 'btn wow fadeInUp');
                    $this->add_render_attribute('tg-button-arg', 'data-wow-delay', '.6s');
                }
            }

        ?>

            <section class="banner-area tg-section-area">
                <div class="container">
                    <div class="row align-items-center justify-content-center">
                        <?php if (!empty($settings['banner_img']['url'])) : ?>
                            <div class="col-lg-5 col-md-9 order-0 order-lg-2">
                                <div class="banner-img wow fadeInRight" data-wow-delay=".2s">
                                    <img src="<?php echo esc_url($settings['banner_img']['url']); ?>" alt="<?php echo esc_attr__('Banner Image', 'genixcore') ?>">
                                </div>
                            </div>
                        <?php endif; ?>
                        <div class="col-lg-7">
                            <div class="banner-content tg-content tg-heading-subheading animation-style2">
                                <?php if (!empty($settings['banner_title'])) : ?>
                                    <h2 class="title tg-element-title">
                                        <?php echo genix_kses($settings['banner_title']); ?>
                                    </h2>
                                <?php endif; ?>

                                <?php if (!empty($settings['banner_desc'])) : ?>
                                    <p class="wow fadeInUp" data-wow-delay=".4s"><?php echo genix_kses($settings['banner_desc']); ?></p>
                                <?php endif; ?>

                                <?php if (!empty($settings['banner_button_show'])) : ?>
                                    <a <?php echo $this->get_render_attribute_string('tg-button-arg'); ?>>
                                        <?php echo $settings['banner_btn_text']; ?><span></span>
                                    </a>
                                <?php endif; ?>

                            </div>
                        </div>
                    </div>
                </div>

                <div class="banner-shape-wrap">
                    <?php if (!empty($settings['shape_img']['url'])) : ?>
                        <img data-parallax='{"x" : 50 , "y" : 0 }' src="<?php echo esc_url($settings['shape_img']['url']); ?>" alt="<?php echo esc_attr__('Banner Shape', 'genixcore') ?>">
                    <?php endif; ?>

                    <?php if (!empty($settings['shape_img2']['url'])) : ?>
                        <img src="<?php echo esc_url($settings['shape_img2']['url']); ?>" alt="<?php echo esc_attr__('Banner Shape', 'genixcore') ?>">
                    <?php endif; ?>

                    <?php if (!empty($settings['shape_img3']['url'])) : ?>
                        <img data-parallax='{"x" : 0 , "y" : 80 }' src="<?php echo esc_url($settings['shape_img3']['url']); ?>" alt="<?php echo esc_attr__('Banner Shape', 'genixcore') ?>">
                    <?php endif; ?>
                </div>

            </section>

        <?php endif; ?>

<?php
    }
}

$widgets_manager->register(new TG_Hero_Banner());
