<?php

namespace GenixCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Xotric Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TG_Team extends Widget_Base
{

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tg-team';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Team', 'genixcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'genix-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['genixcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['genixcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {


        // layout Panel
        $this->start_controls_section(
            'genix_layout',
            [
                'label' => esc_html__('Design Layout', 'genixcore'),
            ]
        );
        $this->add_control(
            'genix_design_style',
            [
                'label' => esc_html__('Select Layout', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'genixcore'),
                    'layout-2' => esc_html__('Layout 2', 'genixcore'),
                    'layout-3' => esc_html__('Layout 3', 'genixcore'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();


        // member list
        $this->start_controls_section(
            '_section_teams',
            [
                'label' => esc_html__('Members', 'genixcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'image',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => esc_html__('Team Image', 'genixcore'),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'team_name',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => esc_html__('Team Name', 'genixcore'),
                'default' => esc_html__('Serhii Ivanenko', 'genixcore'),
                'placeholder' => esc_html__('Type name here', 'genixcore'),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'team_url',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'label' => esc_html__('Team URL', 'genixcore'),
                'default' => esc_html__('#', 'genixcore'),
                'placeholder' => esc_html__('Type url here', 'genixcore'),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'designation',
            [
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'show_label' => true,
                'label' => esc_html__('Designation', 'genixcore'),
                'default' => esc_html__('Abdmarket CTO', 'genixcore'),
                'placeholder' => esc_html__('Type designation here', 'genixcore'),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        // REPEATER
        $this->add_control(
            'teams',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'team_name' => esc_html__('Serhii Ivanenko', 'genixcore'),
                        'designation' => esc_html__('Abdmarket CTO', 'genixcore'),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'team_name' => esc_html__('Alay Macdonald', 'genixcore'),
                        'designation' => esc_html__('Marketing Strategist', 'genixcore'),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'team_name' => esc_html__('William Adams', 'genixcore'),
                        'designation' => esc_html__('Marketologist', 'genixcore'),
                    ],
                    [
                        'image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'team_name' => esc_html__('Alexey Sergienko', 'genixcore'),
                        'designation' => esc_html__('Programmer', 'genixcore'),
                    ],
                ],
                'title_field' => '{{{ team_name }}}',
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail',
                'default' => 'medium_large',
                'separator' => 'before',
                'exclude' => [
                    'custom'
                ]
            ]
        );

        $this->end_controls_section();


        // STYLE TAB
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style', 'genixcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'text_transform',
            [
                'label' => esc_html__('Text Transform', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => esc_html__('None', 'genixcore'),
                    'uppercase' => esc_html__('UPPERCASE', 'genixcore'),
                    'lowercase' => esc_html__('lowercase', 'genixcore'),
                    'capitalize' => esc_html__('Capitalize', 'genixcore'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $xotric_rtl_check = is_rtl() ?  'true' : 'false';

    ?>


        <!-- style 2 -->
        <?php if ($settings['genix_design_style'] === 'layout-2') : ?>

            <script>
                jQuery(document).ready(function($) {

                    /*===========================================
                        =    Team Active		    =
                    =============================================*/
                    $('.team-active').slick({
                        dots: false,
                        infinite: true,
                        speed: 1000,
                        autoplay: true,
                        rtl: <?php echo $xotric_rtl_check; ?>,
                        arrows: false,
                        slidesToShow: 4,
                        slidesToScroll: 1,
                        responsive: [{
                                breakpoint: 1200,
                                settings: {
                                    slidesToShow: 3,
                                    slidesToScroll: 1,
                                    infinite: true,
                                }
                            },
                            {
                                breakpoint: 992,
                                settings: {
                                    slidesToShow: 2,
                                    slidesToScroll: 1
                                }
                            },
                            {
                                breakpoint: 767,
                                settings: {
                                    slidesToShow: 1,
                                    slidesToScroll: 1,
                                    arrows: false,
                                }
                            },
                            {
                                breakpoint: 575,
                                settings: {
                                    slidesToShow: 1,
                                    slidesToScroll: 1,
                                    arrows: false,
                                }
                            },
                        ]
                    });

                });
            </script>

            <div class="row team-active">
                <?php foreach ($settings['teams'] as $item) :

                    if (!empty($item['image']['url'])) {
                        $genix_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url($item['image']['id'], $settings['thumbnail_size']) : $item['image']['url'];
                        $genix_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                    }
                ?>
                    <div class="col-xl-3 col-lg-4 col-md-6 col-sm-10">
                        <div class="team-item">
                            <?php if (!empty($genix_team_image_url)) : ?>
                                <div class="team-thumb">
                                    <a href="<?php echo esc_url($item['team_url']); ?>"><img src="<?php echo esc_url($genix_team_image_url); ?>" alt="<?php echo esc_attr($genix_team_image_alt); ?>"></a>
                                </div>
                            <?php endif; ?>
                            <div class="team-content">
                                <h2 class="title"><a href="<?php echo esc_url($item['team_url']); ?>"><?php echo genix_kses($item['team_name']); ?></a></h2>
                                <?php if (!empty($item['designation'])) : ?>
                                    <span><?php echo genix_kses($item['designation']); ?></span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>


        <!-- style 2 -->
        <?php elseif ($settings['genix_design_style'] === 'layout-3') : ?>

            <script>
                jQuery(document).ready(function($) {

                    /*===========================================
                        =    New Team Active		    =
                    =============================================*/
                    $('.new-team-active').slick({
                        dots: false,
                        infinite: true,
                        speed: 1000,
                        autoplay: true,
                        rtl: <?php echo $xotric_rtl_check; ?>,
                        arrows: false,
                        slidesToShow: 4,
                        slidesToScroll: 1,
                        responsive: [{
                                breakpoint: 1200,
                                settings: {
                                    slidesToShow: 3,
                                    slidesToScroll: 1,
                                    infinite: true,
                                }
                            },
                            {
                                breakpoint: 992,
                                settings: {
                                    slidesToShow: 2,
                                    slidesToScroll: 1
                                }
                            },
                            {
                                breakpoint: 767,
                                settings: {
                                    slidesToShow: 1,
                                    slidesToScroll: 1,
                                    arrows: false,
                                }
                            },
                            {
                                breakpoint: 575,
                                settings: {
                                    slidesToShow: 1,
                                    slidesToScroll: 1,
                                    arrows: false,
                                }
                            },
                        ]
                    });

                });
            </script>

            <div class="row new-team-active">
                <?php foreach ($settings['teams'] as $item) :

                    if (!empty($item['image']['url'])) {
                        $genix_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url($item['image']['id'], $settings['thumbnail_size']) : $item['image']['url'];
                        $genix_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                    }
                ?>
                    <div class="col-xl-3 col-lg-4 col-md-6 col-sm-10">
                        <div class="new-team-item">
                            <?php if (!empty($genix_team_image_url)) : ?>
                                <div class="new-team-thumb">
                                    <a href="<?php echo esc_url($item['team_url']); ?>"><img src="<?php echo esc_url($genix_team_image_url); ?>" alt="<?php echo esc_attr($genix_team_image_alt); ?>"></a>
                                </div>
                            <?php endif; ?>
                            <div class="new-team-content">
                                <h2 class="title"><a href="<?php echo esc_url($item['team_url']); ?>"><?php echo genix_kses($item['team_name']); ?></a></h2>
                                <?php if (!empty($item['designation'])) : ?>
                                    <span><?php echo genix_kses($item['designation']); ?></span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

        <?php else : ?>

            <div class="row justify-content-center">
                <?php foreach ($settings['teams'] as $item) :

                    if (!empty($item['image']['url'])) {
                        $genix_team_image_url = !empty($item['image']['id']) ? wp_get_attachment_image_url($item['image']['id'], $settings['thumbnail_size']) : $item['image']['url'];
                        $genix_team_image_alt = get_post_meta($item["image"]["id"], "_wp_attachment_image_alt", true);
                    }
                ?>
                    <div class="col-xl-3 col-lg-4 col-md-6 col-sm-10">
                        <div class="team-item">
                            <?php if (!empty($genix_team_image_url)) : ?>
                                <div class="team-thumb">
                                    <a href="<?php echo esc_url($item['team_url']); ?>"><img src="<?php echo esc_url($genix_team_image_url); ?>" alt="<?php echo esc_attr($genix_team_image_alt); ?>"></a>
                                </div>
                            <?php endif; ?>
                            <div class="team-content">
                                <h2 class="title"><a href="<?php echo esc_url($item['team_url']); ?>"><?php echo genix_kses($item['team_name']); ?></a></h2>
                                <?php if (!empty($item['designation'])) : ?>
                                    <span><?php echo genix_kses($item['designation']); ?></span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>

        <?php endif; ?>

<?php
    }
}

$widgets_manager->register(new TG_Team());
