<?php

namespace GenixCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;
use \Elementor\Utils;
use \Elementor\Control_Media;

if (!defined('ABSPATH')) exit; // Exit if accessed directly

/**
 * Xotric Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TG_CTA extends Widget_Base
{

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tg-cta';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('CTA', 'genixcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'genix-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['genixcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['genixcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {

        // layout Panel
        $this->start_controls_section(
            'genix_layout',
            [
                'label' => esc_html__('Design Layout', 'genixcore'),
            ]
        );
        $this->add_control(
            'tg_design_style',
            [
                'label' => esc_html__('Select Layout', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'genixcore'),
                    'layout-2' => esc_html__('Layout 2', 'genixcore'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        // _tg_title
        $this->start_controls_section(
            '_tg_title_text',
            [
                'label' => esc_html__('Content', 'genixcore'),
                'condition' => [
                    'tg_design_style' => 'layout-1'
                ]
            ]
        );

        $this->add_control(
            'sub_title',
            [
                'label' => esc_html__('Sub Title', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Lets Work Together', 'genixcore'),
                'placeholder' => esc_html__('Type Text Here', 'genixcore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'tg_title',
            [
                'label' => esc_html__('Title', 'genixcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Need a Successful Your Project?', 'genixcore'),
                'placeholder' => esc_html__('Type Text Here', 'genixcore'),
                'label_block' => true,
            ]
        );

        $this->end_controls_section();

        // tg_button_group
        $this->start_controls_section(
            'tg_button_group',
            [
                'label' => esc_html__('Button', 'genixcore'),
                'condition' => [
                    'tg_design_style' => 'layout-1'
                ]
            ]
        );

        $this->add_control(
            'tg_button_show',
            [
                'label' => esc_html__('Show Button', 'genixcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'genixcore'),
                'label_off' => esc_html__('Hide', 'genixcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'tg_btn_text',
            [
                'label' => esc_html__('Button Text', 'genixcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Estimate Project', 'genixcore'),
                'title' => esc_html__('Enter button text', 'genixcore'),
                'label_block' => true,
                'condition' => [
                    'tg_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'tg_btn_link_type',
            [
                'label' => esc_html__('Button Link Type', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    '1' => 'Custom Link',
                    '2' => 'Internal Page',
                ],
                'default' => '1',
                'label_block' => true,
                'condition' => [
                    'tg_button_show' => 'yes'
                ],
            ]
        );

        $this->add_control(
            'tg_btn_link',
            [
                'label' => esc_html__('Button link', 'genixcore'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => esc_html__('https://your-link.com', 'genixcore'),
                'show_external' => false,
                'default' => [
                    'url' => '#',
                    'is_external' => true,
                    'nofollow' => true,
                    'custom_attributes' => '',
                ],
                'condition' => [
                    'tg_btn_link_type' => '1',
                    'tg_button_show' => 'yes'
                ],
                'label_block' => true,
            ]
        );
        $this->add_control(
            'tg_btn_page_link',
            [
                'label' => esc_html__('Select Button Page', 'genixcore'),
                'type' => Controls_Manager::SELECT2,
                'label_block' => true,
                'options' => genix_get_all_pages(),
                'condition' => [
                    'tg_btn_link_type' => '2',
                    'tg_button_show' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();

        // _tg_image
        $this->start_controls_section(
            '_tg_images',
            [
                'label' => esc_html__('Images', 'genixcore'),
                'condition' => [
                    'tg_design_style' => 'layout-1'
                ]
            ]
        );

        $this->add_control(
            'tg_image',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => esc_html__('Upload Image', 'genixcore'),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->end_controls_section();

        // _tg_text
        $this->start_controls_section(
            '_tg_cta_repeater',
            [
                'label' => esc_html__('Contact Info', 'genixcore'),
                'condition' => [
                    'tg_design_style' => 'layout-2'
                ]
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'tg_image_icon',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => esc_html__('Upload Icon', 'genixcore'),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $repeater->add_control(
            'info_sub_title',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('Sub Info', 'genixcore'),
                'default' => esc_html__('Mail To Us', 'genixcore'),
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater->add_control(
            'info_title',
            [
                'type' => Controls_Manager::TEXT,
                'label' => esc_html__('Info Title', 'genixcore'),
                'default' => esc_html__('Envato@gmail.com', 'genixcore'),
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'list_items',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'info_sub_title' => esc_html__('Mail To Us', 'genixcore'),
                        'info_title' => esc_html__('Envato@gmail.com', 'genixcore'),
                    ],
                    [
                        'info_sub_title' => esc_html__('Call for help:', 'genixcore'),
                        'info_title' => esc_html__('(+055) 6489 6448', 'genixcore'),
                    ],
                    [
                        'info_sub_title' => esc_html__('Address', 'genixcore'),
                        'info_title' => esc_html__('380 Albert St, New York, USA', 'genixcore'),
                    ],
                ],
                'title_field' => '{{ info_title }}',
            ]
        );

        $this->end_controls_section();

        // Style Tab
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Style', 'genixcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'text_transform',
            [
                'label' => esc_html__('Text Transform', 'genixcore'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => esc_html__('None', 'genixcore'),
                    'uppercase' => esc_html__('UPPERCASE', 'genixcore'),
                    'lowercase' => esc_html__('lowercase', 'genixcore'),
                    'capitalize' => esc_html__('Capitalize', 'genixcore'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .title' => 'text-transform: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display(); ?>


        <?php if ($settings['tg_design_style']  == 'layout-2') : ?>

            <div class="contact-wrap wow fadeInUp" data-wow-delay=".2s">
                <div class="contact-shape">
                    <svg viewBox="0 0 1169 220" fill="none" xmlns="http://www.w3.org/2000/svg" preserveAspectRatio="none">
                        <path d="M0 88.4997C0 72.5288 12.5124 59.3578 28.4622 58.5391L1137.46 1.6187C1154.61 0.738463 1169 14.4067 1169 31.5793V190C1169 206.569 1155.57 220 1139 220H30C13.4315 220 0 206.569 0 190V88.4997Z" fill="currentcolor" />
                    </svg>
                </div>
                <div class="contact-item-wrap">
                    <ul class="list-wrap">
                        <?php foreach ($settings['list_items'] as $item) : ?>
                            <li>
                                <div class="contact-item">
                                    <?php if (!empty($item['tg_image_icon']['url'])) : ?>
                                        <div class="icon">
                                            <img src="<?php echo esc_url($item['tg_image_icon']['url']) ?>" alt="<?php echo esc_attr__('Icon', 'genixcore') ?>">
                                        </div>
                                    <?php endif; ?>
                                    <div class="content">
                                        <?php if (!empty($item['info_sub_title'])) : ?>
                                            <span><?php echo esc_html($item['info_sub_title']); ?></span>
                                        <?php endif; ?>

                                        <?php if (!empty($item['info_title'])) : ?>
                                            <a href="<?php echo esc_html($item['info_title']); ?>">
                                                <?php echo esc_html($item['info_title']); ?>
                                            </a>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            </div>

        <?php else :

            // Link
            if ('2' == $settings['tg_btn_link_type']) {
                $this->add_render_attribute('tg-button-arg', 'href', get_permalink($settings['tg_btn_page_link']));
                $this->add_render_attribute('tg-button-arg', 'target', '_self');
                $this->add_render_attribute('tg-button-arg', 'rel', 'nofollow');
                $this->add_render_attribute('tg-button-arg', 'class', 'btn');
            } else {
                if (!empty($settings['tg_btn_link']['url'])) {
                    $this->add_link_attributes('tg-button-arg', $settings['tg_btn_link']);
                    $this->add_render_attribute('tg-button-arg', 'class', 'btn');
                }
            }
        ?>

            <div class="newsletter-inner">
                <div class="newsletter-content">
                    <div class="section-title white-title mb-30 tg-heading-subheading animation-style2">
                        <?php if (!empty($settings['sub_title'])) : ?>
                            <span class="sub-title tg-element-title">
                                <?php echo genix_kses($settings['sub_title']); ?>
                            </span>
                        <?php endif; ?>

                        <?php if (!empty($settings['tg_title'])) : ?>
                            <h2 class="title tg-element-title"><?php echo genix_kses($settings['tg_title']); ?></h2>
                        <?php endif; ?>
                    </div>

                    <?php if (!empty($settings['tg_button_show'])) : ?>
                        <a <?php echo $this->get_render_attribute_string('tg-button-arg'); ?>><?php echo $settings['tg_btn_text']; ?> <span></span></a>
                    <?php endif; ?>

                </div>

                <?php if (!empty($settings['tg_image']['url'])) : ?>
                    <div class="newsletter-img">
                        <img src="<?php echo esc_url($settings['tg_image']['url']) ?>" alt="<?php echo esc_attr__('Image', 'genixcore') ?>" class="wow fadeInRight" data-wow-delay=".2s">
                    </div>
                <?php endif; ?>

            </div>

        <?php endif; ?>

<?php
    }
}

$widgets_manager->register(new TG_CTA());
